<?php
header('Content-Type: application/json');
require_once '../config/config.php';

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON data
$data = json_decode(file_get_contents('php://input'), true);

// Validate required fields
$required = ['product_id', 'product_name', 'product_price', 'quantity', 'customer_name', 'customer_phone', 'delivery_address'];
foreach ($required as $field) {
    if (empty($data[$field])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => "Missing required field: $field"]);
        exit;
    }
}

try {
    // Start transaction
    $pdo->beginTransaction();
    
    // Check if customer exists by phone
    $stmt = $pdo->prepare("SELECT id FROM customers WHERE phone = ?");
    $stmt->execute([$data['customer_phone']]);
    $customer = $stmt->fetch();
    
    $customer_id = null;
    
    // If customer doesn't exist, create new customer
    if (!$customer) {
        $stmt = $pdo->prepare("INSERT INTO customers (name, phone, email, address) VALUES (?, ?, ?, ?)");
        $stmt->execute([
            $data['customer_name'],
            $data['customer_phone'],
            $data['customer_email'] ?? '',
            $data['delivery_address']
        ]);
        $customer_id = $pdo->lastInsertId();
    } else {
        $customer_id = $customer['id'];
        
        // Update customer info if provided
        $stmt = $pdo->prepare("UPDATE customers SET name = ?, email = ?, address = ? WHERE id = ?");
        $stmt->execute([
            $data['customer_name'],
            $data['customer_email'] ?? '',
            $data['delivery_address'],
            $customer_id
        ]);
    }
    
    // Calculate total
    $total_amount = floatval($data['product_price']) * intval($data['quantity']);
    
    // Insert order
    $stmt = $pdo->prepare("INSERT INTO shop_orders 
        (customer_id, customer_name, customer_phone, customer_email, delivery_address, 
         product_id, product_name, product_price, quantity, total_amount, status) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')");
    
    $stmt->execute([
        $customer_id,
        $data['customer_name'],
        $data['customer_phone'],
        $data['customer_email'] ?? '',
        $data['delivery_address'],
        $data['product_id'],
        $data['product_name'],
        $data['product_price'],
        $data['quantity'],
        $total_amount
    ]);
    
    $order_id = $pdo->lastInsertId();
    
    // Update product quantity
    $stmt = $pdo->prepare("UPDATE products SET quantity = quantity - ? WHERE id = ? AND quantity >= ?");
    $stmt->execute([
        $data['quantity'],
        $data['product_id'],
        $data['quantity']
    ]);
    
    if ($stmt->rowCount() === 0) {
        throw new Exception('Insufficient stock');
    }
    
    // Commit transaction
    $pdo->commit();
    
    // Send notification for new order
    notifyNewOrder($pdo, $order_id, $data['customer_name']);
    
    // Check for low stock and notify
    $stmt = $pdo->prepare("SELECT quantity, reorder_level FROM products WHERE id = ?");
    $stmt->execute([$data['product_id']]);
    $product = $stmt->fetch();
    
    if ($product && $product['quantity'] <= $product['reorder_level']) {
        notifyLowStock($pdo, $data['product_id'], $data['product_name'], $product['quantity']);
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Order placed successfully',
        'order_id' => $order_id,
        'customer_id' => $customer_id
    ]);
    
} catch (Exception $e) {
    // Rollback on error
    $pdo->rollBack();
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to place order: ' . $e->getMessage()
    ]);
}
