// Modern Product Management System - JavaScript with SweetAlert2

// Sound effects using Web Audio API
const AudioContext = window.AudioContext || window.webkitAudioContext;
const audioContext = new AudioContext();

// Create sound effects
function playClickSound() {
    const oscillator = audioContext.createOscillator();
    const gainNode = audioContext.createGain();
    
    oscillator.connect(gainNode);
    gainNode.connect(audioContext.destination);
    
    oscillator.frequency.value = 800;
    oscillator.type = 'sine';
    
    gainNode.gain.setValueAtTime(0.1, audioContext.currentTime);
    gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.1);
    
    oscillator.start(audioContext.currentTime);
    oscillator.stop(audioContext.currentTime + 0.1);
}

function playNotificationSound() {
    const oscillator = audioContext.createOscillator();
    const gainNode = audioContext.createGain();
    
    oscillator.connect(gainNode);
    gainNode.connect(audioContext.destination);
    
    oscillator.frequency.value = 600;
    oscillator.type = 'sine';
    
    gainNode.gain.setValueAtTime(0.15, audioContext.currentTime);
    gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.3);
    
    oscillator.start(audioContext.currentTime);
    oscillator.stop(audioContext.currentTime + 0.3);
    
    // Second tone
    setTimeout(() => {
        const osc2 = audioContext.createOscillator();
        const gain2 = audioContext.createGain();
        
        osc2.connect(gain2);
        gain2.connect(audioContext.destination);
        
        osc2.frequency.value = 800;
        osc2.type = 'sine';
        
        gain2.gain.setValueAtTime(0.15, audioContext.currentTime);
        gain2.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.3);
        
        osc2.start(audioContext.currentTime);
        osc2.stop(audioContext.currentTime + 0.3);
    }, 100);
}

function playSuccessSound() {
    const oscillator = audioContext.createOscillator();
    const gainNode = audioContext.createGain();
    
    oscillator.connect(gainNode);
    gainNode.connect(audioContext.destination);
    
    oscillator.frequency.value = 523.25; // C5
    oscillator.type = 'sine';
    
    gainNode.gain.setValueAtTime(0.2, audioContext.currentTime);
    gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.4);
    
    oscillator.start(audioContext.currentTime);
    oscillator.stop(audioContext.currentTime + 0.4);
    
    // Second note
    setTimeout(() => {
        const osc2 = audioContext.createOscillator();
        const gain2 = audioContext.createGain();
        
        osc2.connect(gain2);
        gain2.connect(audioContext.destination);
        
        osc2.frequency.value = 659.25; // E5
        osc2.type = 'sine';
        
        gain2.gain.setValueAtTime(0.2, audioContext.currentTime);
        gain2.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
        
        osc2.start(audioContext.currentTime);
        osc2.stop(audioContext.currentTime + 0.5);
    }, 150);
}

document.addEventListener('DOMContentLoaded', function() {
    
    // Initialize sidebar toggle
    initSidebarToggle();
    
    // Initialize user dropdown
    initUserDropdown();
    
    // Initialize animations
    initAnimations();
    
    // Handle flash messages with SweetAlert2
    handleFlashMessages();
    
    // Add click sounds to all buttons and nav links
    addClickSounds();
    
    // Auto-dismiss regular alerts after 5 seconds
    const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
    alerts.forEach(alert => {
        setTimeout(() => {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });
    
    // Confirm delete actions with SweetAlert2
    const deleteLinks = document.querySelectorAll('a[href*="delete.php"]');
    deleteLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const href = this.href;
            
            Swal.fire({
                title: 'Are you sure?',
                text: "You won't be able to revert this!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#033a71',
                cancelButtonColor: '#6c757d',
                confirmButtonText: '<i class="fas fa-trash-alt"></i> Yes, delete it!',
                cancelButtonText: '<i class="fas fa-times"></i> Cancel',
                reverseButtons: true,
                customClass: {
                    confirmButton: 'btn btn-danger me-2',
                    cancelButton: 'btn btn-secondary'
                },
                buttonsStyling: false
            }).then((result) => {
                if (result.isConfirmed) {
                    // Show loading
                    Swal.fire({
                        title: 'Deleting...',
                        text: 'Please wait',
                        allowOutsideClick: false,
                        allowEscapeKey: false,
                        showConfirmButton: false,
                        willOpen: () => {
                            Swal.showLoading();
                        }
                    });
                    
                    // Redirect to delete URL
                    window.location.href = href;
                }
            });
        });
    });
    
    // Form validation with enhanced feedback
    const forms = document.querySelectorAll('.needs-validation');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!form.checkValidity()) {
                e.preventDefault();
                e.stopPropagation();
                
                Swal.fire({
                    icon: 'error',
                    title: 'Validation Error',
                    text: 'Please fill in all required fields correctly.',
                    confirmButtonColor: '#033a71',
                    customClass: {
                        confirmButton: 'btn btn-primary'
                    },
                    buttonsStyling: false
                });
            }
            form.classList.add('was-validated');
        });
    });
    
    // Number input validation
    const numberInputs = document.querySelectorAll('input[type="number"]');
    numberInputs.forEach(input => {
        input.addEventListener('input', function() {
            if (this.value < 0) {
                this.value = 0;
            }
        });
    });
    
    // Search functionality for tables
    const searchInput = document.getElementById('tableSearch');
    if (searchInput) {
        searchInput.addEventListener('keyup', function() {
            const filter = this.value.toLowerCase();
            const table = document.querySelector('table tbody');
            const rows = table.getElementsByTagName('tr');
            
            let visibleCount = 0;
            for (let i = 0; i < rows.length; i++) {
                const row = rows[i];
                const text = row.textContent.toLowerCase();
                if (text.includes(filter)) {
                    row.style.display = '';
                    visibleCount++;
                } else {
                    row.style.display = 'none';
                }
            }
            
            // Show message if no results
            if (visibleCount === 0 && filter !== '') {
                if (!document.getElementById('noResults')) {
                    const noResults = document.createElement('tr');
                    noResults.id = 'noResults';
                    noResults.innerHTML = '<td colspan="100%" class="text-center py-4"><i class="fas fa-search"></i> No results found</td>';
                    table.appendChild(noResults);
                }
            } else {
                const noResults = document.getElementById('noResults');
                if (noResults) {
                    noResults.remove();
                }
            }
        });
    }
    
    // Print functionality
    const printButtons = document.querySelectorAll('.btn-print');
    printButtons.forEach(btn => {
        btn.addEventListener('click', function() {
            window.print();
        });
    });
    
    // Tooltip initialization
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Format currency inputs
    const currencyInputs = document.querySelectorAll('.currency-input');
    currencyInputs.forEach(input => {
        input.addEventListener('blur', function() {
            if (this.value) {
                this.value = parseFloat(this.value).toFixed(2);
            }
        });
    });
    
    // Add active class to current nav item
    highlightActiveNav();
    
    // Initialize smooth scrolling
    initSmoothScroll();
    
    // Initialize dark mode toggle
    initDarkMode();
    
});

// Add click sounds to buttons and navigation
function addClickSounds() {
    // Add sound to all buttons
    document.querySelectorAll('button, .btn, .sidebar-link, .nav-link, .header-icon-btn').forEach(element => {
        element.addEventListener('click', function(e) {
            // Resume audio context on first user interaction
            if (audioContext.state === 'suspended') {
                audioContext.resume();
            }
            playClickSound();
        });
    });
    
    // Add sound to form submissions
    document.querySelectorAll('form').forEach(form => {
        form.addEventListener('submit', function(e) {
            if (audioContext.state === 'suspended') {
                audioContext.resume();
            }
            playClickSound();
        });
    });
}

// Initialize sidebar toggle
function initSidebarToggle() {
    const toggleBtn = document.getElementById('toggleSidebar');
    const sidebar = document.getElementById('sidebar');
    const mainContent = document.querySelector('.main-content');
    
    if (toggleBtn && sidebar) {
        toggleBtn.addEventListener('click', function() {
            if (window.innerWidth > 992) {
                sidebar.classList.toggle('collapsed');
                mainContent.classList.toggle('expanded');
            } else {
                sidebar.classList.toggle('show');
            }
        });
    }
    
    // Close sidebar on mobile when clicking outside
    document.addEventListener('click', function(e) {
        if (window.innerWidth <= 992) {
            if (!sidebar.contains(e.target) && !toggleBtn.contains(e.target)) {
                sidebar.classList.remove('show');
            }
        }
    });
}

// Initialize user dropdown
function initUserDropdown() {
    const userBtn = document.getElementById('userProfileBtn');
    const userMenu = document.getElementById('userDropdownMenu');
    
    if (userBtn && userMenu) {
        userBtn.addEventListener('click', function(e) {
            e.stopPropagation();
            userMenu.classList.toggle('show');
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', function(e) {
            if (!userBtn.contains(e.target) && !userMenu.contains(e.target)) {
                userMenu.classList.remove('show');
            }
        });
    }
}

// Handle flash messages with SweetAlert2
function handleFlashMessages() {
    const flashAlert = document.querySelector('.modern-alert[data-flash-message]');
    if (flashAlert) {
        const message = flashAlert.getAttribute('data-flash-message');
        const type = flashAlert.getAttribute('data-flash-type');
        
        let icon = 'info';
        if (type === 'success') icon = 'success';
        else if (type === 'danger') icon = 'error';
        else if (type === 'warning') icon = 'warning';
        
        // Hide the regular alert
        flashAlert.style.display = 'none';
        
        // Show SweetAlert2 toast
        const Toast = Swal.mixin({
            toast: true,
            position: 'top-end',
            showConfirmButton: false,
            timer: 4000,
            timerProgressBar: true,
            didOpen: (toast) => {
                toast.addEventListener('mouseenter', Swal.stopTimer);
                toast.addEventListener('mouseleave', Swal.resumeTimer);
            }
        });
        
        Toast.fire({
            icon: icon,
            title: message
        });
    }
}

// Initialize animations
function initAnimations() {
    // Add fade-in animation to cards
    const cards = document.querySelectorAll('.card');
    cards.forEach((card, index) => {
        card.style.animationDelay = `${index * 0.1}s`;
        card.classList.add('fade-in');
    });
}

// Highlight active navigation item
function highlightActiveNav() {
    const currentPath = window.location.pathname;
    const navLinks = document.querySelectorAll('.modern-navbar .nav-link');
    
    navLinks.forEach(link => {
        const href = link.getAttribute('href');
        if (currentPath.includes(href) && href !== 'index.php') {
            link.classList.add('active');
        } else if (currentPath.endsWith('index.php') && href === 'index.php') {
            link.classList.add('active');
        }
    });
}

// Initialize smooth scrolling
function initSmoothScroll() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
}

// Initialize dark mode toggle
function initDarkMode() {
    const darkModeBtn = document.querySelector('.header-icon-btn[title="Toggle Dark Mode"]');
    
    if (!darkModeBtn) return;
    
    // Check for saved dark mode preference
    const isDarkMode = localStorage.getItem('darkMode') === 'true';
    if (isDarkMode) {
        document.body.classList.add('dark-mode');
        darkModeBtn.querySelector('i').classList.replace('fa-moon', 'fa-sun');
    }
    
    darkModeBtn.addEventListener('click', function() {
        document.body.classList.toggle('dark-mode');
        const icon = this.querySelector('i');
        
        if (document.body.classList.contains('dark-mode')) {
            icon.classList.replace('fa-moon', 'fa-sun');
            localStorage.setItem('darkMode', 'true');
            
            const Toast = Swal.mixin({
                toast: true,
                position: 'top-end',
                showConfirmButton: false,
                timer: 2000,
                timerProgressBar: true
            });
            
            Toast.fire({
                icon: 'success',
                title: 'Dark mode enabled'
            });
        } else {
            icon.classList.replace('fa-sun', 'fa-moon');
            localStorage.setItem('darkMode', 'false');
            
            const Toast = Swal.mixin({
                toast: true,
                position: 'top-end',
                showConfirmButton: false,
                timer: 2000,
                timerProgressBar: true
            });
            
            Toast.fire({
                icon: 'success',
                title: 'Light mode enabled'
            });
        }
    });
}

// Utility function to format currency
function formatCurrency(amount) {
    return '$' + parseFloat(amount).toFixed(2);
}

// Utility function to show loading spinner
function showLoading(element) {
    const originalText = element.innerHTML;
    element.setAttribute('data-original-text', originalText);
    element.innerHTML = '<span class="spinner-border spinner-border-sm me-2" role="status"></span> Loading...';
    element.disabled = true;
}

// Utility function to hide loading spinner
function hideLoading(element) {
    const originalText = element.getAttribute('data-original-text');
    element.innerHTML = originalText;
    element.disabled = false;
}

// Show success message with SweetAlert2
function showSuccess(title, message) {
    Swal.fire({
        icon: 'success',
        title: title,
        text: message,
        confirmButtonColor: '#76ba3f',
        customClass: {
            confirmButton: 'btn btn-success'
        },
        buttonsStyling: false
    });
}

// Show error message with SweetAlert2
function showError(title, message) {
    Swal.fire({
        icon: 'error',
        title: title,
        text: message,
        confirmButtonColor: '#033a71',
        customClass: {
            confirmButton: 'btn btn-primary'
        },
        buttonsStyling: false
    });
}

// Show warning message with SweetAlert2
function showWarning(title, message) {
    Swal.fire({
        icon: 'warning',
        title: title,
        text: message,
        confirmButtonColor: '#033a71',
        customClass: {
            confirmButton: 'btn btn-primary'
        },
        buttonsStyling: false
    });
}

// Show info message with SweetAlert2
function showInfo(title, message) {
    Swal.fire({
        icon: 'info',
        title: title,
        text: message,
        confirmButtonColor: '#033a71',
        customClass: {
            confirmButton: 'btn btn-primary'
        },
        buttonsStyling: false
    });
}

// Confirm action with SweetAlert2
function confirmAction(title, message, confirmText = 'Yes', cancelText = 'No') {
    return Swal.fire({
        title: title,
        text: message,
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#033a71',
        cancelButtonColor: '#6c757d',
        confirmButtonText: confirmText,
        cancelButtonText: cancelText,
        reverseButtons: true,
        customClass: {
            confirmButton: 'btn btn-primary me-2',
            cancelButton: 'btn btn-secondary'
        },
        buttonsStyling: false
    });
}

// Show loading overlay
function showLoadingOverlay(message = 'Loading...') {
    Swal.fire({
        title: message,
        allowOutsideClick: false,
        allowEscapeKey: false,
        showConfirmButton: false,
        willOpen: () => {
            Swal.showLoading();
        }
    });
}

// Hide loading overlay
function hideLoadingOverlay() {
    Swal.close();
}

// Format number with commas
function formatNumber(num) {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

// Debounce function for search
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Copy to clipboard with feedback
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        const Toast = Swal.mixin({
            toast: true,
            position: 'top-end',
            showConfirmButton: false,
            timer: 2000,
            timerProgressBar: true
        });
        
        Toast.fire({
            icon: 'success',
            title: 'Copied to clipboard!'
        });
    }).catch(() => {
        showError('Error', 'Failed to copy to clipboard');
    });
}

// Export table to CSV
function exportTableToCSV(tableId, filename = 'export.csv') {
    const table = document.getElementById(tableId);
    if (!table) return;
    
    let csv = [];
    const rows = table.querySelectorAll('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = [];
        const cols = rows[i].querySelectorAll('td, th');
        
        for (let j = 0; j < cols.length; j++) {
            row.push(cols[j].innerText);
        }
        
        csv.push(row.join(','));
    }
    
    // Download CSV
    const csvFile = new Blob([csv.join('\n')], { type: 'text/csv' });
    const downloadLink = document.createElement('a');
    downloadLink.download = filename;
    downloadLink.href = window.URL.createObjectURL(csvFile);
    downloadLink.style.display = 'none';
    document.body.appendChild(downloadLink);
    downloadLink.click();
    document.body.removeChild(downloadLink);
    
    showSuccess('Success', 'Table exported successfully!');
}


// ============================================
// Real-time Notification System
// ============================================

let notificationInterval;
let currentNotifications = [];

function loadNotifications() {
    fetch(BASE_URL + '/api/get-notifications.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const previousCount = currentNotifications.filter(n => n.is_read == 0).length;
                const newCount = data.unread_count;
                
                // Play notification sound if there are new notifications
                if (newCount > previousCount && previousCount >= 0) {
                    if (audioContext.state === 'suspended') {
                        audioContext.resume();
                    }
                    playNotificationSound();
                }
                
                updateNotificationBadge(data.unread_count);
                currentNotifications = data.notifications;
            }
        })
        .catch(error => console.error('Error loading notifications:', error));
}

function updateNotificationBadge(count) {
    const badge = document.querySelector('.notification-badge');
    if (badge) {
        badge.textContent = count;
        badge.style.display = count > 0 ? 'flex' : 'none';
    }
}

function markAsRead(notificationId) {
    playClickSound();
    fetch(BASE_URL + '/api/mark-notification-read.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ notification_id: notificationId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            playSuccessSound();
            loadNotifications();
            // Refresh the modal
            setTimeout(() => {
                showNotificationsModal();
            }, 300);
        }
    })
    .catch(error => console.error('Error marking notification as read:', error));
}

function markAllAsRead() {
    playClickSound();
    fetch(BASE_URL + '/api/mark-notification-read.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ notification_id: 'all' })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            playSuccessSound();
            loadNotifications();
            Swal.fire({
                toast: true,
                position: 'top-end',
                icon: 'success',
                title: 'All notifications marked as read',
                showConfirmButton: false,
                timer: 2000
            });
            // Refresh the modal after a short delay
            setTimeout(() => {
                showNotificationsModal();
            }, 300);
        }
    })
    .catch(error => console.error('Error marking all as read:', error));
}

// Notification Bell Click Handler
document.addEventListener('DOMContentLoaded', function() {
    const notificationBell = document.getElementById('notificationBell');
    
    if (notificationBell) {
        notificationBell.addEventListener('click', function() {
            showNotificationsModal();
        });
    }
    
    // Load notifications on page load
    loadNotifications();
    
    // Refresh notifications every 30 seconds
    notificationInterval = setInterval(loadNotifications, 30000);
});

// Show notifications modal
function showNotificationsModal() {
    playClickSound();
    if (currentNotifications.length === 0) {
        Swal.fire({
            title: '<strong>Notifications</strong>',
            html: '<div style="text-align: center; padding: 2rem;"><i class="fas fa-bell-slash fa-3x text-muted mb-3"></i><p class="text-muted">No notifications</p></div>',
            showConfirmButton: false,
            showCloseButton: true,
            customClass: {
                popup: 'notifications-popup'
            }
        });
        return;
    }
    
    let html = '<div style="max-height: 400px; overflow-y: auto;">';
    
    currentNotifications.forEach(notif => {
        const iconColors = {
            'shopping-bag': '#033a71',
            'exclamation-triangle': '#ffc107',
            'info-circle': '#17a2b8',
            'file-invoice': '#76ba3f',
            'money-bill-wave': '#28a745'
        };
        
        const iconColor = iconColors[notif.icon] || '#6c757d';
        const unreadDot = notif.is_read == 0 ? '<span style="width: 8px; height: 8px; background: #033a71; border-radius: 50%; display: inline-block; margin-right: 8px;"></span>' : '';
        const bgColor = notif.is_read == 0 ? 'rgba(3, 58, 113, 0.05)' : 'transparent';
        
        html += `
            <div class="notification-item" style="padding: 1rem; border-bottom: 1px solid #e3e6f0; background: ${bgColor}; transition: all 0.3s;" 
                 onmouseover="this.style.background='#f8f9fa'" 
                 onmouseout="this.style.background='${bgColor}'">
                <div style="display: flex; align-items: start; gap: 1rem;">
                    <div style="flex-shrink: 0;">
                        ${unreadDot}
                        <i class="fas fa-${notif.icon}" style="font-size: 1.5rem; color: ${iconColor};"></i>
                    </div>
                    <div style="flex: 1;">
                        <strong style="font-size: 1rem; display: block; margin-bottom: 0.25rem;">${notif.title}</strong>
                        <p style="margin: 0 0 0.25rem 0; color: #6c757d; font-size: 0.9rem;">${notif.message} <span style="color: #adb5bd;">(${formatNotificationTime(notif.created_at)})</span></p>
                        <div style="display: flex; gap: 0.5rem; margin-top: 0.5rem;">
                            ${notif.link ? `<button onclick="window.viewNotificationLink('${notif.link}')" class="btn btn-sm btn-primary" style="border-radius: 20px; padding: 0.25rem 0.75rem;">
                                <i class="fas fa-eye"></i> View
                            </button>` : ''}
                            ${notif.is_read == 0 ? `<button onclick="window.markNotificationAsRead(${notif.id})" class="btn btn-sm btn-success" style="border-radius: 20px; padding: 0.25rem 0.75rem;">
                                <i class="fas fa-check"></i> Mark as read
                            </button>` : ''}
                        </div>
                    </div>
                </div>
            </div>
        `;
    });
    
    html += '</div>';
    
    Swal.fire({
        title: '<div style="display: flex; justify-content: space-between; align-items: center; width: 100%; text-align: left;"><strong>Notifications</strong><button onclick="window.markAllNotificationsAsRead()" class="btn btn-sm" style="background: none; border: none; color: #033a71; font-weight: 600; cursor: pointer;"><i class="fas fa-check-double"></i> Mark all as read</button></div>',
        html: html,
        showConfirmButton: false,
        showCloseButton: true,
        width: '600px',
        customClass: {
            popup: 'notifications-popup',
            closeButton: 'notifications-close-btn',
            title: 'notifications-title',
            htmlContainer: 'notifications-content'
        }
    });
}

// Global function to mark notification as read
window.markNotificationAsRead = function(notificationId) {
    markAsRead(notificationId);
};

// Global function to mark all as read
window.markAllNotificationsAsRead = function() {
    markAllAsRead();
};

// Global function to view notification link
window.viewNotificationLink = function(link) {
    if (link) {
        Swal.close();
        window.location.href = link;
    }
};

function handleNotificationClick(id, link, isRead) {
    if (isRead == 0) {
        markAsRead(id);
    }
    
    if (link) {
        Swal.close();
        window.location.href = link;
    }
}

function formatNotificationTime(datetime) {
    const date = new Date(datetime);
    const now = new Date();
    const diff = Math.floor((now - date) / 1000); // seconds
    
    if (diff < 60) return 'Just now';
    if (diff < 3600) return Math.floor(diff / 60) + ' min ago';
    if (diff < 86400) return Math.floor(diff / 3600) + ' hours ago';
    if (diff < 604800) return Math.floor(diff / 86400) + ' days ago';
    
    return date.toLocaleDateString();
}
