// Direct Purchase with Cart Option
class DirectPurchase {
    constructor() {
        this.cart = this.loadCart();
        this.init();
    }

    init() {
        this.updateCartBadge();
        this.attachEventListeners();
    }

    loadCart() {
        const cart = localStorage.getItem('shopping_cart');
        return cart ? JSON.parse(cart) : [];
    }

    saveCart() {
        localStorage.setItem('shopping_cart', JSON.stringify(this.cart));
        this.updateCartBadge();
    }

    updateCartBadge() {
        const badge = document.getElementById('cartBadge');
        if (badge) {
            const count = this.cart.reduce((sum, item) => sum + item.quantity, 0);
            badge.textContent = count;
            badge.style.display = count > 0 ? 'flex' : 'none';
        }
    }

    addToCart(product, quantity = 1) {
        const existingItem = this.cart.find(item => item.id === product.id);
        
        if (existingItem) {
            if (existingItem.quantity + quantity <= product.stock) {
                existingItem.quantity += quantity;
            } else {
                this.showNotification('Maximum stock reached', 'warning');
                return false;
            }
        } else {
            this.cart.push({
                id: product.id,
                name: product.name,
                price: product.price,
                image: product.image,
                quantity: quantity,
                stock: product.stock
            });
        }
        
        this.saveCart();
        return true;
    }

    attachEventListeners() {
        // Buy Now buttons - Show popup with cart option
        document.querySelectorAll('.btn-buy-now').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const product = {
                    id: btn.dataset.id,
                    name: btn.dataset.name,
                    price: parseFloat(btn.dataset.price),
                    image: btn.dataset.image,
                    stock: parseInt(btn.dataset.stock)
                };
                this.showPurchaseForm(product);
            });
        });
    }

    showPurchaseForm(product) {
        const imageSrc = product.image ? `uploads/${product.image}` : '';
        const imageHtml = product.image ? 
            `<img src="${imageSrc}" alt="${product.name}" style="width: 100%; height: 200px; object-fit: cover; border-radius: 12px; margin-bottom: 1.5rem;">` :
            `<div style="width: 100%; height: 200px; background: linear-gradient(135deg, #f5f7fa 0%, #e9ecef 100%); border-radius: 12px; display: flex; align-items: center; justify-content: center; margin-bottom: 1.5rem;"><i class="fas fa-image" style="font-size: 4rem; color: #6c757d; opacity: 0.3;"></i></div>`;
        
        Swal.fire({
            title: '<strong style="color: #033a71;">Complete Your Purchase</strong>',
            html: `
                <div style="text-align: left;">
                    ${imageHtml}
                    <div style="background: linear-gradient(135deg, #f5f7fa 0%, #e9ecef 100%); padding: 1.5rem; border-radius: 12px; margin-bottom: 1.5rem;">
                        <h3 style="margin: 0 0 0.5rem 0; color: #033a71; font-size: 1.3rem;">${product.name}</h3>
                        <p style="margin: 0; font-size: 1.75rem; font-weight: 800; color: #76ba3f;">₦${product.price.toFixed(2)}</p>
                    </div>
                    
                    <div style="margin-bottom: 1rem;">
                        <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #2c3e50;">
                            <i class="fas fa-box"></i> Quantity *
                        </label>
                        <input type="number" id="quantity" class="swal2-input" value="1" min="1" max="${product.stock}" style="width: 100%; margin: 0;">
                        <small style="color: #6c757d;">Available: ${product.stock} units</small>
                    </div>
                    
                    <div style="margin-bottom: 1rem;">
                        <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #2c3e50;">
                            <i class="fas fa-user"></i> Full Name *
                        </label>
                        <input type="text" id="customerName" class="swal2-input" placeholder="Enter your full name" style="width: 100%; margin: 0;">
                    </div>
                    
                    <div style="margin-bottom: 1rem;">
                        <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #2c3e50;">
                            <i class="fas fa-phone"></i> Phone Number *
                        </label>
                        <input type="tel" id="customerPhone" class="swal2-input" placeholder="e.g., 08012345678" style="width: 100%; margin: 0;">
                    </div>
                    
                    <div style="margin-bottom: 1rem;">
                        <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #2c3e50;">
                            <i class="fas fa-envelope"></i> Email (Optional)
                        </label>
                        <input type="email" id="customerEmail" class="swal2-input" placeholder="your@email.com" style="width: 100%; margin: 0;">
                    </div>
                    
                    <div style="margin-bottom: 1rem;">
                        <label style="display: block; margin-bottom: 0.5rem; font-weight: 600; color: #2c3e50;">
                            <i class="fas fa-map-marker-alt"></i> Delivery Address *
                        </label>
                        <textarea id="customerAddress" class="swal2-textarea" placeholder="Enter your delivery address" style="width: 100%; margin: 0; min-height: 80px;"></textarea>
                    </div>
                </div>
            `,
            width: '600px',
            showCancelButton: true,
            showDenyButton: true,
            confirmButtonText: '<i class="fas fa-check-circle"></i> Buy Now',
            denyButtonText: '<i class="fas fa-cart-plus"></i> Add to Cart',
            cancelButtonText: '<i class="fas fa-times"></i> Cancel',
            confirmButtonColor: '#76ba3f',
            denyButtonColor: '#033a71',
            cancelButtonColor: '#6c757d',
            customClass: {
                popup: 'animated-popup',
                confirmButton: 'btn-confirm-order',
                denyButton: 'btn-add-to-cart',
                cancelButton: 'btn-cancel-order'
            },
            showClass: {
                popup: 'animate__animated animate__fadeInDown animate__faster'
            },
            hideClass: {
                popup: 'animate__animated animate__fadeOutUp animate__faster'
            },
            preConfirm: () => {
                const name = document.getElementById('customerName').value.trim();
                const phone = document.getElementById('customerPhone').value.trim();
                const email = document.getElementById('customerEmail').value.trim();
                const address = document.getElementById('customerAddress').value.trim();
                const quantity = parseInt(document.getElementById('quantity').value);
                
                if (!name || !phone || !address) {
                    Swal.showValidationMessage('<i class="fas fa-exclamation-circle"></i> Please fill in all required fields');
                    return false;
                }
                
                if (quantity < 1 || quantity > product.stock) {
                    Swal.showValidationMessage(`<i class="fas fa-exclamation-circle"></i> Quantity must be between 1 and ${product.stock}`);
                    return false;
                }
                
                const total = product.price * quantity;
                
                return { name, phone, email, address, quantity, total };
            },
            preDeny: () => {
                const quantity = parseInt(document.getElementById('quantity').value);
                
                if (quantity < 1 || quantity > product.stock) {
                    Swal.showValidationMessage(`<i class="fas fa-exclamation-circle"></i> Quantity must be between 1 and ${product.stock}`);
                    return false;
                }
                
                return { quantity };
            }
        }).then((result) => {
            if (result.isConfirmed) {
                // Buy Now - Direct purchase
                const orderData = result.value;
                this.processOrder(product, orderData);
            } else if (result.isDenied) {
                // Add to Cart
                const cartData = result.value;
                if (this.addToCart(product, cartData.quantity)) {
                    this.showNotification(`Added ${cartData.quantity} item(s) to cart!`, 'success');
                }
            }
        });
    }

    processOrder(product, orderData) {
        // Show loading
        Swal.fire({
            title: 'Processing Order...',
            html: '<i class="fas fa-spinner fa-spin fa-3x" style="color: #033a71;"></i>',
            showConfirmButton: false,
            allowOutsideClick: false
        });
        
        // Send order to server
        fetch('api/place-order.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                product_id: product.id,
                product_name: product.name,
                product_price: product.price,
                quantity: orderData.quantity,
                customer_name: orderData.name,
                customer_phone: orderData.phone,
                customer_email: orderData.email,
                delivery_address: orderData.address
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                Swal.fire({
                    icon: 'success',
                    title: '<strong style="color: #76ba3f;">Order Placed Successfully!</strong>',
                    html: `
                        <div style="text-align: left; background: #f8f9fa; padding: 1.5rem; border-radius: 12px; margin-top: 1rem;">
                            <p style="margin: 0.5rem 0;"><strong>Order ID:</strong> #${data.order_id}</p>
                            <p style="margin: 0.5rem 0;"><strong>Product:</strong> ${product.name}</p>
                            <p style="margin: 0.5rem 0;"><strong>Quantity:</strong> ${orderData.quantity}</p>
                            <p style="margin: 0.5rem 0;"><strong>Total:</strong> <span style="color: #76ba3f; font-size: 1.5rem; font-weight: 800;">₦${orderData.total.toFixed(2)}</span></p>
                            <p style="margin: 0.5rem 0;"><strong>Delivery to:</strong> ${orderData.address}</p>
                            <hr style="margin: 1rem 0; border-color: #dee2e6;">
                            <p style="margin: 0.5rem 0; color: #6c757d;"><i class="fas fa-info-circle"></i> We will contact you at <strong>${orderData.phone}</strong> to confirm your order.</p>
                        </div>
                    `,
                    confirmButtonText: '<i class="fas fa-shopping-bag"></i> Continue Shopping',
                    confirmButtonColor: '#033a71',
                    customClass: {
                        popup: 'animated-popup'
                    }
                }).then(() => {
                    // Reload page to update stock
                    window.location.reload();
                });
            } else {
                throw new Error(data.message || 'Failed to place order');
            }
        })
        .catch(error => {
            Swal.fire({
                icon: 'error',
                title: 'Order Failed',
                text: error.message || 'Something went wrong. Please try again.',
                confirmButtonColor: '#dc3545'
            });
        });
    }

    showNotification(message, type = 'success') {
        const icon = type === 'success' ? 'success' : type === 'warning' ? 'warning' : 'info';
        Swal.fire({
            toast: true,
            position: 'top-end',
            icon: icon,
            title: message,
            showConfirmButton: false,
            timer: 3000,
            timerProgressBar: true,
            customClass: {
                popup: 'animated-toast'
            }
        });
    }
}

// Initialize direct purchase
const purchase = new DirectPurchase();

// Dark Mode Toggle
document.addEventListener('DOMContentLoaded', function() {
    const darkModeBtn = document.getElementById('darkModeToggle');
    
    if (darkModeBtn) {
        // Check for saved dark mode preference
        const isDarkMode = localStorage.getItem('shopDarkMode') === 'true';
        if (isDarkMode) {
            document.body.classList.add('dark-mode');
            darkModeBtn.querySelector('i').classList.replace('fa-moon', 'fa-sun');
        }
        
        darkModeBtn.addEventListener('click', function() {
            document.body.classList.toggle('dark-mode');
            const icon = this.querySelector('i');
            
            if (document.body.classList.contains('dark-mode')) {
                icon.classList.replace('fa-moon', 'fa-sun');
                localStorage.setItem('shopDarkMode', 'true');
                
                Swal.fire({
                    toast: true,
                    position: 'top-end',
                    icon: 'success',
                    title: 'Dark mode enabled',
                    showConfirmButton: false,
                    timer: 2000,
                    timerProgressBar: true
                });
            } else {
                icon.classList.replace('fa-sun', 'fa-moon');
                localStorage.setItem('shopDarkMode', 'false');
                
                Swal.fire({
                    toast: true,
                    position: 'top-end',
                    icon: 'success',
                    title: 'Light mode enabled',
                    showConfirmButton: false,
                    timer: 2000,
                    timerProgressBar: true
                });
            }
        });
    }
    
    // Add fade-in animation to cards
    const cards = document.querySelectorAll('.card, .col-md-4, .col-lg-3');
    cards.forEach((card, index) => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(20px)';
        card.style.transition = 'all 0.5s ease-out';
        
        setTimeout(() => {
            card.style.opacity = '1';
            card.style.transform = 'translateY(0)';
        }, index * 100);
    });
    
    // Add hover animation to product cards
    document.querySelectorAll('.product-card-hover').forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-10px) scale(1.02)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
});
