<?php
// Core utility functions

/**
 * Sanitize input data
 */
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate CSRF token
 */
function validateCSRF($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Redirect to a page
 */
function redirect($page) {
    header("Location: $page");
    exit();
}

/**
 * Set flash message
 */
function setFlash($type, $message) {
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

/**
 * Get and clear flash message
 */
function getFlash() {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

/**
 * Format currency
 */
function formatCurrency($amount) {
    return CURRENCY . number_format($amount, 2);
}

/**
 * Format date
 */
function formatDate($date, $format = 'Y-m-d H:i:s') {
    return date($format, strtotime($date));
}

/**
 * Generate invoice number
 */
function generateInvoiceNumber($pdo) {
    $prefix = 'INV';
    $date = date('Ymd');
    
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM invoices WHERE DATE(created_at) = CURDATE()");
    $stmt->execute();
    $count = $stmt->fetchColumn() + 1;
    
    return $prefix . '-' . $date . '-' . str_pad($count, 4, '0', STR_PAD_LEFT);
}

/**
 * Check low stock products
 */
function getLowStockProducts($pdo) {
    $stmt = $pdo->query("SELECT * FROM products WHERE quantity <= reorder_level ORDER BY quantity ASC");
    return $stmt->fetchAll();
}

/**
 * Calculate outstanding balance
 */
function getOutstandingBalance($invoiceId, $pdo) {
    $stmt = $pdo->prepare("SELECT grand_total, amount_paid FROM invoices WHERE id = ?");
    $stmt->execute([$invoiceId]);
    $invoice = $stmt->fetch();
    
    if ($invoice) {
        return $invoice['grand_total'] - $invoice['amount_paid'];
    }
    return 0;
}

/**
 * Update product stock
 */
function updateProductStock($productId, $quantity, $type, $pdo, $referenceType = null, $referenceId = null) {
    global $pdo;
    
    // Update product quantity
    if ($type === 'out') {
        $stmt = $pdo->prepare("UPDATE products SET quantity = quantity - ? WHERE id = ?");
    } else {
        $stmt = $pdo->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
    }
    $stmt->execute([$quantity, $productId]);
    
    // Record stock movement
    $stmt = $pdo->prepare("INSERT INTO stock_movements (product_id, type, quantity, reference_type, reference_id, created_by) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->execute([$productId, $type, $quantity, $referenceType, $referenceId, $_SESSION['user_id'] ?? null]);
}

/**
 * Get sales statistics
 */
function getSalesStats($pdo, $period = 'today') {
    $where = '';
    
    switch ($period) {
        case 'today':
            $where = "DATE(created_at) = CURDATE()";
            break;
        case 'week':
            $where = "YEARWEEK(created_at) = YEARWEEK(NOW())";
            break;
        case 'month':
            $where = "MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())";
            break;
        default:
            $where = "1=1";
    }
    
    $stmt = $pdo->query("SELECT 
        COUNT(*) as total_invoices,
        SUM(grand_total) as total_revenue,
        SUM(CASE WHEN payment_status = 'paid' THEN grand_total ELSE 0 END) as paid_amount,
        SUM(CASE WHEN payment_status = 'unpaid' THEN grand_total ELSE 0 END) as unpaid_amount
        FROM invoices WHERE $where");
    
    return $stmt->fetch();
}

/**
 * Get best selling products
 */
function getBestSellingProducts($pdo, $limit = 10) {
    $stmt = $pdo->prepare("SELECT p.name, p.sku, SUM(ii.quantity) as total_sold, SUM(ii.total) as revenue
        FROM invoice_items ii
        JOIN products p ON ii.product_id = p.id
        GROUP BY ii.product_id
        ORDER BY total_sold DESC
        LIMIT ?");
    $stmt->execute([$limit]);
    return $stmt->fetchAll();
}
