<?php
/**
 * Notification Helper Functions
 */

/**
 * Add a notification
 * @param PDO $pdo Database connection
 * @param int|null $user_id User ID (null for all users)
 * @param string $type Notification type (order, stock, invoice, etc.)
 * @param string $title Notification title
 * @param string $message Notification message
 * @param string|null $link Link to related page
 * @param string $icon Font Awesome icon name
 */
function addNotification($pdo, $user_id, $type, $title, $message, $link = null, $icon = 'bell') {
    try {
        $stmt = $pdo->prepare("INSERT INTO notifications (user_id, type, title, message, link, icon) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$user_id, $type, $title, $message, $link, $icon]);
        return true;
    } catch (Exception $e) {
        error_log("Failed to add notification: " . $e->getMessage());
        return false;
    }
}

/**
 * Add notification for new order
 */
function notifyNewOrder($pdo, $order_id, $customer_name) {
    addNotification(
        $pdo,
        null, // All users
        'order',
        'New Order Received',
        "New order from {$customer_name}",
        "modules/orders/view.php?id={$order_id}",
        'shopping-bag'
    );
}

/**
 * Add notification for low stock
 */
function notifyLowStock($pdo, $product_id, $product_name, $quantity) {
    addNotification(
        $pdo,
        null, // All users
        'stock',
        'Low Stock Alert',
        "{$product_name} is running low (Only {$quantity} left)",
        "modules/products/edit.php?id={$product_id}",
        'exclamation-triangle'
    );
}

/**
 * Add notification for order status change
 */
function notifyOrderStatus($pdo, $order_id, $status, $customer_name) {
    $statusMessages = [
        'confirmed' => 'Order confirmed',
        'processing' => 'Order is being processed',
        'delivered' => 'Order delivered',
        'cancelled' => 'Order cancelled'
    ];
    
    $message = $statusMessages[$status] ?? 'Order status updated';
    
    addNotification(
        $pdo,
        null,
        'order',
        'Order Status Updated',
        "Order from {$customer_name} - {$message}",
        "modules/orders/view.php?id={$order_id}",
        'info-circle'
    );
}

/**
 * Add notification for new invoice
 */
function notifyNewInvoice($pdo, $invoice_id, $invoice_number, $customer_name) {
    addNotification(
        $pdo,
        null,
        'invoice',
        'Invoice Generated',
        "Invoice {$invoice_number} created for {$customer_name}",
        "modules/invoices/view.php?id={$invoice_id}",
        'file-invoice'
    );
}

/**
 * Add notification for payment received
 */
function notifyPaymentReceived($pdo, $invoice_id, $invoice_number, $amount) {
    addNotification(
        $pdo,
        null,
        'payment',
        'Payment Received',
        "Payment of " . formatCurrency($amount) . " received for {$invoice_number}",
        "modules/invoices/view.php?id={$invoice_id}",
        'money-bill-wave'
    );
}
