<?php
require_once '../../config/config.php';
requireLogin();

$pageTitle = 'Create Invoice';

// Get customers
$stmt = $pdo->query("SELECT * FROM customers ORDER BY name");
$customers = $stmt->fetchAll();

// Get products
$stmt = $pdo->query("SELECT * FROM products WHERE quantity > 0 ORDER BY name");
$products = $stmt->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRF($_POST['csrf_token'])) {
        setFlash('danger', 'Invalid request');
        redirect('create.php');
    }
    
    try {
        $pdo->beginTransaction();
        
        $customer_id = $_POST['customer_id'] ?: null;
        $subtotal = floatval($_POST['subtotal']);
        $discount = floatval($_POST['discount']);
        $tax = floatval($_POST['tax']);
        $grand_total = floatval($_POST['grand_total']);
        $payment_method = $_POST['payment_method'];
        $amount_paid = floatval($_POST['amount_paid']);
        $notes = sanitize($_POST['notes']);
        
        // Determine payment status
        if ($amount_paid >= $grand_total) {
            $payment_status = 'paid';
        } elseif ($amount_paid > 0) {
            $payment_status = 'partial';
        } else {
            $payment_status = 'unpaid';
        }
        
        // Generate invoice number
        $invoice_number = generateInvoiceNumber($pdo);
        
        // Insert invoice
        $stmt = $pdo->prepare("INSERT INTO invoices (invoice_number, customer_id, user_id, subtotal, discount, tax, grand_total, payment_method, payment_status, amount_paid, notes) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$invoice_number, $customer_id, $_SESSION['user_id'], $subtotal, $discount, $tax, $grand_total, $payment_method, $payment_status, $amount_paid, $notes]);
        
        $invoice_id = $pdo->lastInsertId();
        
        // Insert invoice items and update stock
        $product_ids = $_POST['product_id'];
        $quantities = $_POST['quantity'];
        $unit_prices = $_POST['unit_price'];
        
        foreach ($product_ids as $index => $product_id) {
            if (empty($product_id)) continue;
            
            $quantity = intval($quantities[$index]);
            $unit_price = floatval($unit_prices[$index]);
            $total = $quantity * $unit_price;
            
            // Get product name
            $stmt = $pdo->prepare("SELECT name FROM products WHERE id = ?");
            $stmt->execute([$product_id]);
            $product = $stmt->fetch();
            
            // Insert invoice item
            $stmt = $pdo->prepare("INSERT INTO invoice_items (invoice_id, product_id, product_name, quantity, unit_price, total) 
                VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$invoice_id, $product_id, $product['name'], $quantity, $unit_price, $total]);
            
            // Update stock
            updateProductStock($product_id, $quantity, 'out', $pdo, 'invoice', $invoice_id);
        }
        
        // Record payment if amount paid
        if ($amount_paid > 0) {
            $stmt = $pdo->prepare("INSERT INTO payments (invoice_id, amount, payment_method) VALUES (?, ?, ?)");
            $stmt->execute([$invoice_id, $amount_paid, $payment_method]);
        }
        
        $pdo->commit();
        
        setFlash('success', 'Invoice created successfully');
        redirect('view.php?id=' . $invoice_id);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = 'Error: ' . $e->getMessage();
    }
}

include '../../includes/header.php';
?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2><i class="fas fa-file-invoice-dollar me-2"></i>Create New Invoice</h2>
        <p class="text-muted mb-0">Fill in the details below to create a new invoice</p>
    </div>
    <a href="list.php" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to Invoices</a>
</div>

<div class="card">
    <div class="card-body">
        <form method="POST" id="invoiceForm">
            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
            
            <!-- Customer Selection Section -->
            <div class="card bg-light mb-4">
                <div class="card-body">
                    <h5 class="mb-3"><i class="fas fa-user-tie me-2 text-primary"></i>Customer Information</h5>
                    <div class="row">
                        <div class="col-md-8 mb-3">
                            <label class="form-label">Customer</label>
                            <select name="customer_id" id="customerSelect" class="form-select">
                                <option value="">Walk-in Customer (No customer record)</option>
                                <?php foreach ($customers as $customer): ?>
                                    <option value="<?php echo $customer['id']; ?>" 
                                            data-phone="<?php echo htmlspecialchars($customer['phone'] ?? ''); ?>"
                                            data-email="<?php echo htmlspecialchars($customer['email'] ?? ''); ?>"
                                            data-address="<?php echo htmlspecialchars($customer['address'] ?? ''); ?>">
                                        <?php echo htmlspecialchars($customer['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Select a customer or leave blank for walk-in sales</small>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Payment Method *</label>
                            <select name="payment_method" class="form-select" required>
                                <option value="cash">Cash</option>
                                <option value="transfer">Bank Transfer</option>
                                <option value="pos">POS</option>
                            </select>
                        </div>
                    </div>
                    <div id="customerInfo" class="mt-2" style="display: none;">
                        <div class="alert alert-info mb-0">
                            <div class="row">
                                <div class="col-md-4">
                                    <small class="text-muted d-block">Phone</small>
                                    <strong id="customerPhone">-</strong>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted d-block">Email</small>
                                    <strong id="customerEmail">-</strong>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted d-block">Address</small>
                                    <strong id="customerAddress">-</strong>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <hr class="my-4">
            
            <!-- Invoice Items Section -->
            <div class="card bg-light mb-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="mb-0"><i class="fas fa-list me-2 text-primary"></i>Invoice Items</h5>
                        <button type="button" class="btn btn-success btn-sm" id="addItem">
                            <i class="fas fa-plus-circle"></i> Add Item
                        </button>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-bordered bg-white">
                            <thead class="table-light">
                                <tr>
                                    <th width="40%">Product</th>
                                    <th width="15%">Quantity</th>
                                    <th width="15%">Unit Price</th>
                                    <th width="20%">Total</th>
                                    <th width="10%">Action</th>
                                </tr>
                            </thead>
                            <tbody id="itemsContainer">
                                <tr class="item-row">
                                    <td>
                                        <select name="product_id[]" class="form-select product-select" required>
                                            <option value="">Select Product</option>
                                            <?php foreach ($products as $product): ?>
                                                <option value="<?php echo $product['id']; ?>" 
                                                        data-price="<?php echo $product['price']; ?>"
                                                        data-stock="<?php echo $product['quantity']; ?>">
                                                    <?php echo htmlspecialchars($product['name']); ?> - <?php echo formatCurrency($product['price']); ?> (Stock: <?php echo $product['quantity']; ?>)
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <input type="number" name="quantity[]" class="form-control quantity-input" value="1" min="1" step="1" required>
                                    </td>
                                    <td>
                                        <input type="number" name="unit_price[]" class="form-control price-input" value="0" min="0" step="0.01" required>
                                    </td>
                                    <td>
                                        <input type="text" class="form-control item-total" value="<?php echo CURRENCY; ?>0.00" readonly>
                                    </td>
                                    <td class="text-center">
                                        <button type="button" class="btn btn-danger btn-sm remove-item" title="Remove Item">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <hr class="my-4">
            
            <!-- Notes and Totals Section -->
            <div class="row">
                <div class="col-md-8">
                    <label class="form-label"><i class="fas fa-sticky-note me-2"></i>Notes</label>
                    <textarea name="notes" class="form-control" rows="5" placeholder="Add any additional notes or special instructions..."></textarea>
                </div>
                
                <div class="col-md-4">
                    <div class="card bg-light">
                        <div class="card-body">
                            <input type="hidden" name="subtotal" id="subtotalHidden">
                            <input type="hidden" name="tax" id="taxHidden">
                            <input type="hidden" name="grand_total" id="grandTotalHidden">
                            
                            <div class="d-flex justify-content-between mb-2">
                                <span>Subtotal:</span>
                                <strong id="subtotalDisplay"><?php echo CURRENCY; ?>0.00</strong>
                            </div>
                            <div class="d-flex justify-content-between mb-2 align-items-center">
                                <span>Discount:</span>
                                <input type="number" name="discount" id="discount" class="form-control form-control-sm" style="width: 100px;" value="0" min="0" step="0.01">
                            </div>
                            <div class="d-flex justify-content-between mb-2 align-items-center">
                                <span>Tax (<?php echo TAX_RATE; ?>%):</span>
                                <strong id="taxDisplay"><?php echo CURRENCY; ?>0.00</strong>
                            </div>
                            <hr>
                            <div class="d-flex justify-content-between mb-3">
                                <h5>Grand Total:</h5>
                                <h5 class="text-primary" id="grandTotalDisplay"><?php echo CURRENCY; ?>0.00</h5>
                            </div>
                            <div class="d-flex justify-content-between align-items-center">
                                <span>Amount Paid:</span>
                                <input type="number" name="amount_paid" id="amountPaid" class="form-control form-control-sm" style="width: 120px;" value="0" min="0" step="0.01" required>
                            </div>
                            <small class="text-muted d-block mt-2">Enter amount received from customer</small>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="mt-4 d-flex justify-content-between">
                <a href="list.php" class="btn btn-secondary btn-lg">
                    <i class="fas fa-times"></i> Cancel
                </a>
                <button type="submit" class="btn btn-primary btn-lg">
                    <i class="fas fa-save"></i> Create Invoice
                </button>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const itemsContainer = document.getElementById('itemsContainer');
    const addItemBtn = document.getElementById('addItem');
    const discountInput = document.getElementById('discount');
    const customerSelect = document.getElementById('customerSelect');
    const customerInfo = document.getElementById('customerInfo');
    
    // Customer selection handler
    customerSelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        if (this.value) {
            document.getElementById('customerPhone').textContent = selectedOption.dataset.phone || '-';
            document.getElementById('customerEmail').textContent = selectedOption.dataset.email || '-';
            document.getElementById('customerAddress').textContent = selectedOption.dataset.address || '-';
            customerInfo.style.display = 'block';
        } else {
            customerInfo.style.display = 'none';
        }
    });
    
    // Product options HTML
    const productOptions = `
        <option value="">Select Product</option>
        <?php foreach ($products as $product): ?>
            <option value="<?php echo $product['id']; ?>" 
                    data-price="<?php echo $product['price']; ?>"
                    data-stock="<?php echo $product['quantity']; ?>">
                <?php echo htmlspecialchars($product['name']); ?> - <?php echo formatCurrency($product['price']); ?> (Stock: <?php echo $product['quantity']; ?>)
            </option>
        <?php endforeach; ?>
    `;
    
    // Add new item row
    addItemBtn.addEventListener('click', function() {
        const newRow = document.createElement('tr');
        newRow.className = 'item-row';
        newRow.innerHTML = `
            <td>
                <select name="product_id[]" class="form-select product-select" required>
                    ${productOptions}
                </select>
            </td>
            <td>
                <input type="number" name="quantity[]" class="form-control quantity-input" value="1" min="1" step="1" required>
            </td>
            <td>
                <input type="number" name="unit_price[]" class="form-control price-input" value="0" min="0" step="0.01" required>
            </td>
            <td>
                <input type="text" class="form-control item-total" value="<?php echo CURRENCY; ?>0.00" readonly>
            </td>
            <td class="text-center">
                <button type="button" class="btn btn-danger btn-sm remove-item" title="Remove Item">
                    <i class="fas fa-trash"></i>
                </button>
            </td>
        `;
        itemsContainer.appendChild(newRow);
        attachItemEvents(newRow);
        calculateTotals();
    });
    
    // Attach events to item rows
    function attachItemEvents(row) {
        const productSelect = row.querySelector('.product-select');
        const quantityInput = row.querySelector('.quantity-input');
        const priceInput = row.querySelector('.price-input');
        const removeBtn = row.querySelector('.remove-item');
        
        productSelect.addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const price = selectedOption.getAttribute('data-price');
            const stock = selectedOption.getAttribute('data-stock');
            
            if (price) {
                priceInput.value = price;
                quantityInput.max = stock;
                calculateItemTotal(row);
            }
        });
        
        quantityInput.addEventListener('input', function() {
            const productSelect = row.querySelector('.product-select');
            const selectedOption = productSelect.options[productSelect.selectedIndex];
            const stock = parseInt(selectedOption.getAttribute('data-stock')) || 0;
            
            if (parseInt(this.value) > stock) {
                Swal.fire({
                    icon: 'warning',
                    title: 'Insufficient Stock',
                    text: `Only ${stock} units available in stock`,
                    confirmButtonColor: '#033a71'
                });
                this.value = stock;
            }
            calculateItemTotal(row);
        });
        
        priceInput.addEventListener('input', () => calculateItemTotal(row));
        
        removeBtn.addEventListener('click', function() {
            if (itemsContainer.querySelectorAll('.item-row').length > 1) {
                row.remove();
                calculateTotals();
            } else {
                Swal.fire({
                    icon: 'warning',
                    title: 'Cannot Remove',
                    text: 'At least one item is required',
                    confirmButtonColor: '#033a71'
                });
            }
        });
    }
    
    // Calculate item total
    function calculateItemTotal(row) {
        const quantity = parseFloat(row.querySelector('.quantity-input').value) || 0;
        const price = parseFloat(row.querySelector('.price-input').value) || 0;
        const total = quantity * price;
        row.querySelector('.item-total').value = '<?php echo CURRENCY; ?>' + total.toFixed(2);
        calculateTotals();
    }
    
    // Calculate all totals
    function calculateTotals() {
        let subtotal = 0;
        document.querySelectorAll('.item-row').forEach(row => {
            const quantity = parseFloat(row.querySelector('.quantity-input').value) || 0;
            const price = parseFloat(row.querySelector('.price-input').value) || 0;
            subtotal += quantity * price;
        });
        
        const discount = parseFloat(discountInput.value) || 0;
        const taxRate = <?php echo TAX_RATE; ?>;
        const taxAmount = ((subtotal - discount) * taxRate) / 100;
        const grandTotal = subtotal - discount + taxAmount;
        
        document.getElementById('subtotalDisplay').textContent = '<?php echo CURRENCY; ?>' + subtotal.toFixed(2);
        document.getElementById('taxDisplay').textContent = '<?php echo CURRENCY; ?>' + taxAmount.toFixed(2);
        document.getElementById('grandTotalDisplay').textContent = '<?php echo CURRENCY; ?>' + grandTotal.toFixed(2);
        
        document.getElementById('subtotalHidden').value = subtotal.toFixed(2);
        document.getElementById('taxHidden').value = taxAmount.toFixed(2);
        document.getElementById('grandTotalHidden').value = grandTotal.toFixed(2);
    }
    
    // Attach events to existing rows
    document.querySelectorAll('.item-row').forEach(row => attachItemEvents(row));
    
    // Discount change
    discountInput.addEventListener('input', calculateTotals);
    
    // Initial calculation
    calculateTotals();
    
    // Form validation
    document.getElementById('invoiceForm').addEventListener('submit', function(e) {
        const hasItems = itemsContainer.querySelectorAll('.item-row').length > 0;
        if (!hasItems) {
            e.preventDefault();
            Swal.fire({
                icon: 'error',
                title: 'No Items',
                text: 'Please add at least one item to the invoice',
                confirmButtonColor: '#033a71'
            });
        }
    });
});
</script>

<?php include '../../includes/footer.php'; ?>
