<?php
require_once '../../config/config.php';
requireLogin();

$id = intval($_GET['id']);

// Get invoice
$stmt = $pdo->prepare("SELECT * FROM invoices WHERE id = ?");
$stmt->execute([$id]);
$invoice = $stmt->fetch();

if (!$invoice) {
    setFlash('danger', 'Invoice not found');
    redirect('list.php');
}

// Get invoice items
$stmt = $pdo->prepare("SELECT * FROM invoice_items WHERE invoice_id = ?");
$stmt->execute([$id]);
$items = $stmt->fetchAll();

// Get customers
$customers = $pdo->query("SELECT * FROM customers ORDER BY name")->fetchAll();

// Get products
$products = $pdo->query("SELECT * FROM products ORDER BY name")->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        $customer_id = intval($_POST['customer_id']);
        $notes = sanitize($_POST['notes']);
        $discount = floatval($_POST['discount'] ?? 0);
        $tax = floatval($_POST['tax'] ?? 0);
        
        // Calculate totals
        $subtotal = 0;
        $item_ids = $_POST['item_id'] ?? [];
        $quantities = $_POST['quantity'] ?? [];
        $prices = $_POST['price'] ?? [];
        
        foreach ($quantities as $key => $qty) {
            $subtotal += floatval($qty) * floatval($prices[$key]);
        }
        
        $grand_total = $subtotal - $discount + $tax;
        
        // Update invoice
        $stmt = $pdo->prepare("UPDATE invoices SET 
            customer_id = ?, 
            subtotal = ?, 
            discount = ?, 
            tax = ?, 
            grand_total = ?, 
            notes = ? 
            WHERE id = ?");
        
        $stmt->execute([
            $customer_id,
            $subtotal,
            $discount,
            $tax,
            $grand_total,
            $notes,
            $id
        ]);
        
        // Delete existing items
        $pdo->prepare("DELETE FROM invoice_items WHERE invoice_id = ?")->execute([$id]);
        
        // Insert new items
        $stmt = $pdo->prepare("INSERT INTO invoice_items (invoice_id, product_id, product_name, quantity, unit_price, total) VALUES (?, ?, ?, ?, ?, ?)");
        
        foreach ($item_ids as $key => $product_id) {
            if (!empty($product_id)) {
                $productStmt = $pdo->prepare("SELECT name FROM products WHERE id = ?");
                $productStmt->execute([$product_id]);
                $product_name = $productStmt->fetchColumn();
                
                $quantity = floatval($quantities[$key]);
                $price = floatval($prices[$key]);
                $total = $quantity * $price;
                
                $stmt->execute([$id, $product_id, $product_name, $quantity, $price, $total]);
            }
        }
        
        $pdo->commit();
        
        setFlash('success', 'Invoice updated successfully');
        redirect('view.php?id=' . $id);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        setFlash('danger', 'Error updating invoice: ' . $e->getMessage());
    }
}

$pageTitle = 'Edit Invoice';
include '../../includes/header.php';
?>

<?php if (isset($_SESSION['flash_message'])): ?>
    <div class="alert alert-<?php echo $_SESSION['flash_type']; ?> alert-dismissible fade show">
        <?php 
        echo $_SESSION['flash_message']; 
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2><i class="fas fa-edit me-2"></i>Edit Invoice</h2>
        <p class="text-muted mb-0">Invoice #<?php echo $invoice['invoice_number']; ?></p>
    </div>
    <div>
        <a href="view.php?id=<?php echo $id; ?>" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to Invoice</a>
        <a href="list.php" class="btn btn-outline-secondary"><i class="fas fa-list"></i> All Invoices</a>
    </div>
</div>

<!-- Invoice Info Card -->
<div class="row mb-4">
    <div class="col-md-12">
        <div class="card">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon info me-3">
                                <i class="fas fa-file-invoice"></i>
                            </div>
                            <div>
                                <small class="text-muted d-block">Invoice Number</small>
                                <strong><?php echo $invoice['invoice_number']; ?></strong>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon <?php 
                                echo $invoice['payment_status'] === 'paid' ? 'success' : 
                                    ($invoice['payment_status'] === 'partial' ? 'warning' : 'danger'); 
                            ?> me-3">
                                <i class="fas fa-<?php 
                                    echo $invoice['payment_status'] === 'paid' ? 'check-circle' : 
                                        ($invoice['payment_status'] === 'partial' ? 'clock' : 'exclamation-circle'); 
                                ?>"></i>
                            </div>
                            <div>
                                <small class="text-muted d-block">Payment Status</small>
                                <span class="badge badge-<?php 
                                    echo $invoice['payment_status'] === 'paid' ? 'success' : 
                                        ($invoice['payment_status'] === 'partial' ? 'warning' : 'danger'); 
                                ?>">
                                    <?php echo ucfirst($invoice['payment_status']); ?>
                                </span>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon primary me-3">
                                <i class="fas fa-money-bill-wave"></i>
                            </div>
                            <div>
                                <small class="text-muted d-block">Amount Paid</small>
                                <strong><?php echo formatCurrency($invoice['amount_paid']); ?></strong>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="d-flex align-items-center">
                            <div class="stat-icon <?php echo ($invoice['grand_total'] - $invoice['amount_paid']) > 0 ? 'danger' : 'success'; ?> me-3">
                                <i class="fas fa-balance-scale"></i>
                            </div>
                            <div>
                                <small class="text-muted d-block">Balance Due</small>
                                <strong><?php echo formatCurrency($invoice['grand_total'] - $invoice['amount_paid']); ?></strong>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-body">
        <form method="POST" id="invoiceForm">
            <!-- Customer Selection Section -->
            <div class="card bg-light mb-4">
                <div class="card-body">
                    <h5 class="mb-3"><i class="fas fa-user-tie me-2 text-primary"></i>Customer Information</h5>
                    <div class="row">
                        <div class="col-md-8 mb-3">
                            <label class="form-label">Customer *</label>
                            <select name="customer_id" id="customerSelect" class="form-select" required>
                                <option value="">Select Customer</option>
                                <?php foreach ($customers as $customer): ?>
                                    <option value="<?php echo $customer['id']; ?>" 
                                            data-phone="<?php echo htmlspecialchars($customer['phone'] ?? ''); ?>"
                                            data-email="<?php echo htmlspecialchars($customer['email'] ?? ''); ?>"
                                            data-address="<?php echo htmlspecialchars($customer['address'] ?? ''); ?>"
                                            <?php echo $invoice['customer_id'] == $customer['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($customer['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Payment Method</label>
                            <select name="payment_method" class="form-select">
                                <option value="cash" <?php echo $invoice['payment_method'] === 'cash' ? 'selected' : ''; ?>>Cash</option>
                                <option value="transfer" <?php echo $invoice['payment_method'] === 'transfer' ? 'selected' : ''; ?>>Bank Transfer</option>
                                <option value="pos" <?php echo $invoice['payment_method'] === 'pos' ? 'selected' : ''; ?>>POS</option>
                            </select>
                        </div>
                    </div>
                    <div id="customerInfo" class="mt-2" style="display: none;">
                        <div class="alert alert-info mb-0">
                            <div class="row">
                                <div class="col-md-4">
                                    <small class="text-muted d-block">Phone</small>
                                    <strong id="customerPhone">-</strong>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted d-block">Email</small>
                                    <strong id="customerEmail">-</strong>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted d-block">Address</small>
                                    <strong id="customerAddress">-</strong>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <hr class="my-4">
            
            <!-- Invoice Items Section -->
            <div class="card bg-light mb-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="mb-0"><i class="fas fa-list me-2 text-primary"></i>Invoice Items</h5>
                        <button type="button" class="btn btn-success btn-sm" id="addItem">
                            <i class="fas fa-plus-circle"></i> Add Item
                        </button>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-bordered bg-white">
                            <thead class="table-light">
                                <tr>
                                    <th width="40%">Product</th>
                                    <th width="15%">Quantity</th>
                                    <th width="15%">Unit Price</th>
                                    <th width="20%">Total</th>
                                    <th width="10%">Action</th>
                                </tr>
                            </thead>
                            <tbody id="itemsContainer">
                <?php if (!empty($items)): ?>
                    <?php foreach ($items as $index => $item): ?>
                    <div class="row item-row mb-3">
                        <div class="col-md-5">
                            <label class="form-label">Product</label>
                            <select name="item_id[]" class="form-select product-select" required>
                                <option value="">Select Product</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?php echo $product['id']; ?>" 
                                            data-price="<?php echo $product['price']; ?>"
                                            <?php 
                                            // Try to match by product_id or product_name
                                            $selected = false;
                                            if (isset($item['product_id']) && $item['product_id'] == $product['id']) {
                                                $selected = true;
                                            } elseif ($item['product_name'] == $product['name']) {
                                                $selected = true;
                                            }
                                            echo $selected ? 'selected' : ''; 
                                            ?>>
                                        <?php echo htmlspecialchars($product['name']); ?> - <?php echo formatCurrency($product['price']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Quantity</label>
                            <input type="number" name="quantity[]" class="form-control quantity-input" value="<?php echo $item['quantity']; ?>" min="1" step="0.01" required>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Price</label>
                            <input type="number" name="price[]" class="form-control price-input" value="<?php echo $item['unit_price']; ?>" min="0" step="0.01" required>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Total</label>
                            <input type="text" class="form-control item-total" value="<?php echo formatCurrency($item['total']); ?>" readonly>
                        </div>
                        <div class="col-md-1">
                            <label class="form-label">&nbsp;</label>
                            <button type="button" class="btn btn-danger w-100 remove-item"><i class="fas fa-trash"></i></button>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="row item-row mb-3">
                        <div class="col-md-5">
                            <label class="form-label">Product</label>
                            <select name="item_id[]" class="form-select product-select" required>
                                <option value="">Select Product</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price']; ?>">
                                        <?php echo htmlspecialchars($product['name']); ?> - <?php echo formatCurrency($product['price']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Quantity</label>
                            <input type="number" name="quantity[]" class="form-control quantity-input" value="1" min="1" step="0.01" required>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Price</label>
                            <input type="number" name="price[]" class="form-control price-input" value="0" min="0" step="0.01" required>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Total</label>
                            <input type="text" class="form-control item-total" value="<?php echo CURRENCY; ?>0.00" readonly>
                        </div>
                        <div class="col-md-1">
                            <label class="form-label">&nbsp;</label>
                            <button type="button" class="btn btn-danger w-100 remove-item"><i class="fas fa-trash"></i></button>
                        </div>
                    </div>
                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <hr class="my-4">
            
            <!-- Notes and Totals Section -->
            <div class="row">
                <div class="col-md-8">
                    <label class="form-label">Notes</label>
                    <textarea name="notes" class="form-control" rows="4"><?php echo htmlspecialchars($invoice['notes']); ?></textarea>
                </div>
                
                <div class="col-md-4">
                    <div class="card bg-light">
                        <div class="card-body">
                            <div class="d-flex justify-content-between mb-2">
                                <span>Subtotal:</span>
                                <strong id="subtotalDisplay"><?php echo formatCurrency($invoice['subtotal']); ?></strong>
                            </div>
                            <div class="d-flex justify-content-between mb-2 align-items-center">
                                <span>Discount:</span>
                                <input type="number" name="discount" id="discount" class="form-control form-control-sm" style="width: 100px;" value="<?php echo $invoice['discount']; ?>" min="0" step="0.01">
                            </div>
                            <div class="d-flex justify-content-between mb-2 align-items-center">
                                <span>Tax:</span>
                                <input type="number" name="tax" id="tax" class="form-control form-control-sm" style="width: 100px;" value="<?php echo $invoice['tax']; ?>" min="0" step="0.01">
                            </div>
                            <hr>
                            <div class="d-flex justify-content-between">
                                <h5>Grand Total:</h5>
                                <h5 class="text-primary" id="grandTotalDisplay"><?php echo formatCurrency($invoice['grand_total']); ?></h5>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="mt-4">
                <button type="submit" class="btn btn-primary btn-lg">
                    <i class="fas fa-save"></i> Update Invoice
                </button>
                <a href="view.php?id=<?php echo $id; ?>" class="btn btn-secondary btn-lg">
                    <i class="fas fa-times"></i> Cancel
                </a>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const itemsContainer = document.getElementById('itemsContainer');
    const addItemBtn = document.getElementById('addItem');
    const discountInput = document.getElementById('discount');
    const taxInput = document.getElementById('tax');
    
    // Product options HTML
    const productOptions = `
        <option value="">Select Product</option>
        <?php foreach ($products as $product): ?>
            <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price']; ?>">
                <?php echo htmlspecialchars($product['name']); ?> - <?php echo formatCurrency($product['price']); ?>
            </option>
        <?php endforeach; ?>
    `;
    
    // Add new item row
    addItemBtn.addEventListener('click', function() {
        const newRow = document.createElement('div');
        newRow.className = 'row item-row mb-3';
        newRow.innerHTML = `
            <div class="col-md-5">
                <label class="form-label">Product</label>
                <select name="item_id[]" class="form-select product-select" required>
                    ${productOptions}
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Quantity</label>
                <input type="number" name="quantity[]" class="form-control quantity-input" value="1" min="1" step="0.01" required>
            </div>
            <div class="col-md-2">
                <label class="form-label">Price</label>
                <input type="number" name="price[]" class="form-control price-input" value="0" min="0" step="0.01" required>
            </div>
            <div class="col-md-2">
                <label class="form-label">Total</label>
                <input type="text" class="form-control item-total" value="<?php echo CURRENCY; ?>0.00" readonly>
            </div>
            <div class="col-md-1">
                <label class="form-label">&nbsp;</label>
                <button type="button" class="btn btn-danger w-100 remove-item"><i class="fas fa-trash"></i></button>
            </div>
        `;
        itemsContainer.appendChild(newRow);
        attachItemEvents(newRow);
        calculateTotals();
    });
    
    // Attach events to item rows
    function attachItemEvents(row) {
        const productSelect = row.querySelector('.product-select');
        const quantityInput = row.querySelector('.quantity-input');
        const priceInput = row.querySelector('.price-input');
        const removeBtn = row.querySelector('.remove-item');
        
        productSelect.addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const price = selectedOption.getAttribute('data-price');
            if (price) {
                priceInput.value = price;
                calculateItemTotal(row);
            }
        });
        
        quantityInput.addEventListener('input', () => calculateItemTotal(row));
        priceInput.addEventListener('input', () => calculateItemTotal(row));
        
        removeBtn.addEventListener('click', function() {
            row.remove();
            calculateTotals();
        });
    }
    
    // Calculate item total
    function calculateItemTotal(row) {
        const quantity = parseFloat(row.querySelector('.quantity-input').value) || 0;
        const price = parseFloat(row.querySelector('.price-input').value) || 0;
        const total = quantity * price;
        row.querySelector('.item-total').value = '<?php echo CURRENCY; ?>' + total.toFixed(2);
        calculateTotals();
    }
    
    // Calculate all totals
    function calculateTotals() {
        let subtotal = 0;
        document.querySelectorAll('.item-row').forEach(row => {
            const quantity = parseFloat(row.querySelector('.quantity-input').value) || 0;
            const price = parseFloat(row.querySelector('.price-input').value) || 0;
            subtotal += quantity * price;
        });
        
        const discount = parseFloat(discountInput.value) || 0;
        const tax = parseFloat(taxInput.value) || 0;
        const grandTotal = subtotal - discount + tax;
        
        document.getElementById('subtotalDisplay').textContent = '<?php echo CURRENCY; ?>' + subtotal.toFixed(2);
        document.getElementById('grandTotalDisplay').textContent = '<?php echo CURRENCY; ?>' + grandTotal.toFixed(2);
    }
    
    // Attach events to existing rows
    document.querySelectorAll('.item-row').forEach(row => attachItemEvents(row));
    
    // Discount and tax change
    discountInput.addEventListener('input', calculateTotals);
    taxInput.addEventListener('input', calculateTotals);
    
    // Initial calculation
    calculateTotals();
});
</script>

<?php include '../../includes/footer.php'; ?>
