<?php
require_once '../../config/config.php';
requireLogin();

$id = intval($_GET['id']);
$status = $_GET['status'] ?? '';

// Validate status
$validStatuses = ['pending', 'confirmed', 'processing', 'delivered', 'cancelled'];
if (!in_array($status, $validStatuses)) {
    setFlash('danger', 'Invalid status');
    redirect('list.php');
}

try {
    // Get order details
    $stmt = $pdo->prepare("SELECT * FROM shop_orders WHERE id = ?");
    $stmt->execute([$id]);
    $order = $stmt->fetch();
    
    if (!$order) {
        setFlash('danger', 'Order not found');
        redirect('list.php');
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    // Update order status
    $stmt = $pdo->prepare("UPDATE shop_orders SET status = ? WHERE id = ?");
    $stmt->execute([$status, $id]);
    
    // If confirming order, create invoice automatically
    if ($status === 'confirmed' && $order['status'] === 'pending') {
        // Generate invoice number
        $stmt = $pdo->query("SELECT MAX(CAST(SUBSTRING(invoice_number, 5) AS UNSIGNED)) as max_num FROM invoices WHERE invoice_number LIKE 'INV-%'");
        $result = $stmt->fetch();
        $next_num = ($result['max_num'] ?? 0) + 1;
        $invoice_number = 'INV-' . str_pad($next_num, 6, '0', STR_PAD_LEFT);
        
        // Create invoice
        $stmt = $pdo->prepare("INSERT INTO invoices 
            (invoice_number, customer_id, user_id, subtotal, tax, grand_total, amount_paid, payment_status, notes) 
            VALUES (?, ?, ?, ?, ?, ?, 0, 'unpaid', ?)");
        
        $tax = $order['total_amount'] * (TAX_RATE / 100);
        $grand_total = $order['total_amount'] + $tax;
        $notes = "Auto-generated from Shop Order #" . $order['id'];
        
        $stmt->execute([
            $invoice_number,
            $order['customer_id'],
            $_SESSION['user_id'],
            $order['total_amount'],
            $tax,
            $grand_total,
            $notes
        ]);
        
        $invoice_id = $pdo->lastInsertId();
        
        // Add invoice item
        $stmt = $pdo->prepare("INSERT INTO invoice_items 
            (invoice_id, product_id, product_name, quantity, unit_price, total) 
            VALUES (?, ?, ?, ?, ?, ?)");
        
        $stmt->execute([
            $invoice_id,
            $order['product_id'],
            $order['product_name'],
            $order['quantity'],
            $order['product_price'],
            $order['total_amount']
        ]);
        
        // Update order with invoice reference
        $stmt = $pdo->prepare("UPDATE shop_orders SET notes = CONCAT(COALESCE(notes, ''), '\nInvoice Generated: ', ?) WHERE id = ?");
        $stmt->execute([$invoice_number, $id]);
        
        // Send notifications
        notifyOrderStatus($pdo, $id, $status, $order['customer_name']);
        notifyNewInvoice($pdo, $invoice_id, $invoice_number, $order['customer_name']);
        
        setFlash('success', 'Order confirmed and Invoice ' . $invoice_number . ' created successfully!');
    } else {
        // Send notification for status change
        if ($status !== $order['status']) {
            notifyOrderStatus($pdo, $id, $status, $order['customer_name']);
        }
        
        setFlash('success', 'Order status updated to ' . ucfirst($status));
    }
    
    // Commit transaction
    $pdo->commit();
    
} catch (PDOException $e) {
    // Rollback on error
    $pdo->rollBack();
    setFlash('danger', 'Error updating status: ' . $e->getMessage());
}

redirect('view.php?id=' . $id);
