<?php
require_once '../../config/config.php';
requireLogin();

$pageTitle = 'Order Details';

$id = intval($_GET['id']);

// Get order details
$stmt = $pdo->prepare("SELECT so.*, p.name as product_name_current, p.image as product_image, p.sku
    FROM shop_orders so
    LEFT JOIN products p ON so.product_id = p.id
    WHERE so.id = ?");
$stmt->execute([$id]);
$order = $stmt->fetch();

if (!$order) {
    setFlash('danger', 'Order not found');
    redirect('list.php');
}

// Check if invoice was generated for this order
$invoice = null;
if (strpos($order['notes'], 'Invoice Generated:') !== false) {
    preg_match('/Invoice Generated: (INV-\d+)/', $order['notes'], $matches);
    if (isset($matches[1])) {
        $stmt = $pdo->prepare("SELECT id, invoice_number, payment_status, grand_total FROM invoices WHERE invoice_number = ?");
        $stmt->execute([$matches[1]]);
        $invoice = $stmt->fetch();
    }
}

include '../../includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-shopping-bag me-2"></i>Order #<?php echo $order['id']; ?></h2>
    <a href="list.php" class="btn btn-secondary">
        <i class="fas fa-arrow-left"></i> Back to Orders
    </a>
</div>

<div class="row">
    <!-- Order Details -->
    <div class="col-md-8">
        <div class="card mb-4">
            <div class="card-header">
                <i class="fas fa-info-circle"></i> Order Information
            </div>
            <div class="card-body">
                <div class="row mb-3">
                    <div class="col-md-6">
                        <strong><i class="fas fa-calendar"></i> Order Date:</strong><br>
                        <?php echo date('F d, Y H:i', strtotime($order['order_date'])); ?>
                    </div>
                    <div class="col-md-6">
                        <strong><i class="fas fa-tag"></i> Status:</strong><br>
                        <?php
                        $statusColors = [
                            'pending' => 'warning',
                            'confirmed' => 'info',
                            'processing' => 'primary',
                            'delivered' => 'success',
                            'cancelled' => 'danger'
                        ];
                        $statusIcons = [
                            'pending' => 'clock',
                            'confirmed' => 'check',
                            'processing' => 'cog',
                            'delivered' => 'check-circle',
                            'cancelled' => 'times-circle'
                        ];
                        $color = $statusColors[$order['status']] ?? 'secondary';
                        $icon = $statusIcons[$order['status']] ?? 'question';
                        ?>
                        <span class="badge badge-<?php echo $color; ?>" style="font-size: 1rem; padding: 0.5rem 1rem;">
                            <i class="fas fa-<?php echo $icon; ?>"></i> <?php echo ucfirst($order['status']); ?>
                        </span>
                    </div>
                </div>
                
                <hr>
                
                <!-- Product Details -->
                <h5 class="mb-3"><i class="fas fa-box"></i> Product Details</h5>
                <div class="d-flex gap-3 mb-3">
                    <?php if (!empty($order['product_image']) && file_exists('../../uploads/' . $order['product_image'])): ?>
                        <img src="../../uploads/<?php echo $order['product_image']; ?>" 
                             alt="<?php echo $order['product_name']; ?>" 
                             style="width: 120px; height: 120px; object-fit: cover; border-radius: 12px; border: 2px solid #e3e6f0;">
                    <?php else: ?>
                        <div style="width: 120px; height: 120px; background: linear-gradient(135deg, #f5f7fa 0%, #e9ecef 100%); border-radius: 12px; display: flex; align-items: center; justify-content: center; border: 2px solid #e3e6f0;">
                            <i class="fas fa-image" style="font-size: 3rem; color: #6c757d; opacity: 0.3;"></i>
                        </div>
                    <?php endif; ?>
                    
                    <div>
                        <h4 class="mb-2"><?php echo $order['product_name']; ?></h4>
                        <?php if ($order['sku']): ?>
                            <p class="text-muted mb-1"><strong>SKU:</strong> <?php echo $order['sku']; ?></p>
                        <?php endif; ?>
                        <p class="mb-1"><strong>Unit Price:</strong> <?php echo formatCurrency($order['product_price']); ?></p>
                        <p class="mb-1"><strong>Quantity:</strong> <span class="badge badge-info"><?php echo $order['quantity']; ?></span></p>
                        <p class="mb-0"><strong>Total:</strong> <span style="color: #76ba3f; font-size: 1.5rem; font-weight: 800;"><?php echo formatCurrency($order['total_amount']); ?></span></p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Notes -->
        <?php if ($order['notes']): ?>
        <div class="card">
            <div class="card-header">
                <i class="fas fa-sticky-note"></i> Order Notes
            </div>
            <div class="card-body">
                <?php echo nl2br(htmlspecialchars($order['notes'])); ?>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Invoice Link -->
        <?php if ($invoice): ?>
        <div class="card">
            <div class="card-header bg-success text-white">
                <i class="fas fa-file-invoice"></i> Generated Invoice
            </div>
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h5 class="mb-1"><?php echo $invoice['invoice_number']; ?></h5>
                        <p class="mb-0">
                            <span class="badge badge-<?php echo $invoice['payment_status'] === 'paid' ? 'success' : ($invoice['payment_status'] === 'partial' ? 'warning' : 'danger'); ?>">
                                <?php echo ucfirst($invoice['payment_status']); ?>
                            </span>
                            <strong class="ms-2"><?php echo formatCurrency($invoice['grand_total']); ?></strong>
                        </p>
                    </div>
                    <a href="../invoices/view.php?id=<?php echo $invoice['id']; ?>" class="btn btn-primary">
                        <i class="fas fa-eye"></i> View Invoice
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
    
    <!-- Customer Details -->
    <div class="col-md-4">
        <div class="card mb-4">
            <div class="card-header">
                <i class="fas fa-user"></i> Customer Information
            </div>
            <div class="card-body">
                <p class="mb-2">
                    <strong><i class="fas fa-user text-muted me-2"></i>Name:</strong><br>
                    <?php echo $order['customer_name']; ?>
                </p>
                <p class="mb-2">
                    <strong><i class="fas fa-phone text-muted me-2"></i>Phone:</strong><br>
                    <a href="tel:<?php echo $order['customer_phone']; ?>"><?php echo $order['customer_phone']; ?></a>
                </p>
                <?php if ($order['customer_email']): ?>
                <p class="mb-2">
                    <strong><i class="fas fa-envelope text-muted me-2"></i>Email:</strong><br>
                    <a href="mailto:<?php echo $order['customer_email']; ?>"><?php echo $order['customer_email']; ?></a>
                </p>
                <?php endif; ?>
                <p class="mb-0">
                    <strong><i class="fas fa-map-marker-alt text-muted me-2"></i>Delivery Address:</strong><br>
                    <?php echo nl2br(htmlspecialchars($order['delivery_address'])); ?>
                </p>
            </div>
        </div>
        
        <!-- Actions -->
        <div class="card">
            <div class="card-header">
                <i class="fas fa-cog"></i> Actions
            </div>
            <div class="card-body">
                <?php if ($order['status'] === 'pending'): ?>
                    <a href="update-status.php?id=<?php echo $order['id']; ?>&status=confirmed" 
                       class="btn btn-success btn-block mb-2">
                        <i class="fas fa-check"></i> Confirm Order
                    </a>
                    <a href="update-status.php?id=<?php echo $order['id']; ?>&status=cancelled" 
                       class="btn btn-danger btn-block mb-2"
                       onclick="return confirm('Are you sure you want to cancel this order?')">
                        <i class="fas fa-times"></i> Cancel Order
                    </a>
                <?php elseif ($order['status'] === 'confirmed'): ?>
                    <a href="update-status.php?id=<?php echo $order['id']; ?>&status=processing" 
                       class="btn btn-primary btn-block mb-2">
                        <i class="fas fa-cog"></i> Mark as Processing
                    </a>
                <?php elseif ($order['status'] === 'processing'): ?>
                    <a href="update-status.php?id=<?php echo $order['id']; ?>&status=delivered" 
                       class="btn btn-success btn-block mb-2">
                        <i class="fas fa-check-circle"></i> Mark as Delivered
                    </a>
                <?php endif; ?>
                
                <?php if ($order['status'] !== 'cancelled'): ?>
                <button type="button" class="btn btn-secondary btn-block" data-bs-toggle="modal" data-bs-target="#notesModal">
                    <i class="fas fa-sticky-note"></i> Add Notes
                </button>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Notes Modal -->
<div class="modal fade" id="notesModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-sticky-note"></i> Add Order Notes</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="add-notes.php">
                <div class="modal-body">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                    <div class="mb-3">
                        <label class="form-label">Notes</label>
                        <textarea name="notes" class="form-control" rows="5" placeholder="Add notes about this order..."><?php echo $order['notes']; ?></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Notes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include '../../includes/footer.php'; ?>
