<?php
require_once '../../config/config.php';
requireLogin();

$pageTitle = 'Edit Product - ' . SITE_NAME;

$id = intval($_GET['id']);

// Get product
$stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
$stmt->execute([$id]);
$product = $stmt->fetch();

if (!$product) {
    setFlash('danger', 'Product not found');
    redirect('list.php');
}

// Get categories
$stmt = $pdo->query("SELECT * FROM categories ORDER BY name");
$categories = $stmt->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRF($_POST['csrf_token'])) {
        setFlash('danger', 'Invalid request');
        redirect('edit.php?id=' . $id);
    }
    
    $name = sanitize($_POST['name']);
    $sku = sanitize($_POST['sku']);
    $category_id = $_POST['category_id'];
    $cost_price = floatval($_POST['cost_price']);
    $selling_price = floatval($_POST['selling_price']);
    $quantity = intval($_POST['quantity']);
    $reorder_level = intval($_POST['reorder_level']);
    $description = sanitize($_POST['description']);
    
    // Handle image upload
    $image = $product['image']; // Keep existing image by default
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $max_size = 5 * 1024 * 1024; // 5MB
        
        if (in_array($_FILES['image']['type'], $allowed_types) && $_FILES['image']['size'] <= $max_size) {
            $extension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
            $new_image = uniqid('product_') . '.' . $extension;
            $upload_path = '../../uploads/' . $new_image;
            
            if (move_uploaded_file($_FILES['image']['tmp_name'], $upload_path)) {
                // Delete old image if exists
                if ($product['image'] && file_exists('../../uploads/' . $product['image'])) {
                    unlink('../../uploads/' . $product['image']);
                }
                $image = $new_image;
            } else {
                $error = 'Failed to upload image';
            }
        } else {
            $error = 'Invalid image file. Please upload JPG, PNG, GIF or WebP (max 5MB)';
        }
    }
    
    // Handle image removal
    if (isset($_POST['remove_image']) && $_POST['remove_image'] === '1') {
        if ($product['image'] && file_exists('../../uploads/' . $product['image'])) {
            unlink('../../uploads/' . $product['image']);
        }
        $image = null;
    }
    
    if (!isset($error)) {
        try {
            $stmt = $pdo->prepare("UPDATE products SET name=?, sku=?, category_id=?, cost_price=?, selling_price=?, quantity=?, reorder_level=?, description=?, image=? WHERE id=?");
            $stmt->execute([$name, $sku, $category_id, $cost_price, $selling_price, $quantity, $reorder_level, $description, $image, $id]);
            
            setFlash('success', 'Product updated successfully');
            redirect('list.php');
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    }
}

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-md-8 offset-md-2">
        <div class="card">
            <div class="card-header">
                <h4>Edit Product</h4>
            </div>
            <div class="card-body">
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>
                
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Product Name *</label>
                            <input type="text" name="name" class="form-control" value="<?php echo $product['name']; ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">SKU / Product Code *</label>
                            <input type="text" name="sku" class="form-control" value="<?php echo $product['sku']; ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Category</label>
                            <select name="category_id" class="form-select">
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $cat): ?>
                                    <option value="<?php echo $cat['id']; ?>" <?php echo $cat['id'] == $product['category_id'] ? 'selected' : ''; ?>>
                                        <?php echo $cat['name']; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Quantity in Stock *</label>
                            <input type="number" name="quantity" class="form-control" value="<?php echo $product['quantity']; ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Cost Price *</label>
                            <input type="number" step="0.01" name="cost_price" class="form-control" value="<?php echo $product['cost_price']; ?>" required>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Selling Price *</label>
                            <input type="number" step="0.01" name="selling_price" class="form-control" value="<?php echo $product['selling_price']; ?>" required>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Reorder Level *</label>
                            <input type="number" name="reorder_level" class="form-control" value="<?php echo $product['reorder_level']; ?>" required>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea name="description" class="form-control" rows="3"><?php echo $product['description']; ?></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label"><i class="fas fa-image me-2"></i>Product Image</label>
                        
                        <?php if (!empty($product['image']) && file_exists('../../uploads/' . $product['image'])): ?>
                            <div class="mb-3">
                                <img src="../../uploads/<?php echo $product['image']; ?>" alt="Current Image" class="img-thumbnail" style="max-width: 200px; max-height: 200px;">
                                <div class="form-check mt-2">
                                    <input type="checkbox" name="remove_image" value="1" class="form-check-input" id="removeImage">
                                    <label class="form-check-label" for="removeImage">
                                        <i class="fas fa-trash text-danger"></i> Remove current image
                                    </label>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <input type="file" name="image" class="form-control" accept="image/jpeg,image/png,image/gif,image/webp">
                        <small class="text-muted">Accepted formats: JPG, PNG, GIF, WebP (Max 5MB)</small>
                    </div>
                    
                    <div class="d-flex justify-content-between">
                        <a href="list.php" class="btn btn-secondary">Cancel</a>
                        <button type="submit" class="btn btn-primary">Update Product</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include '../../includes/footer.php'; ?>
